##########################################################################################################################################
#                                                                                                                                        #
# Code to replicate the primary analysis of ecosystem multifunctionality as presented in 
# Ramus et al. (2017) An invasive foundation species enhances multifunctionality in a coastal ecosystem. PNAS 114(32):8580-8585.  
# invFSxfunc code from https://github.com/apramus/invFSxfunc
#                                                                                                                                        #
# This code calculates multifunctionaity, fits candidate models using nonlinear least squares for each function individually, and        #
# generates the model selection table and corresponding figure presented in the paper. 
#
# Using the multifunc package for analysis of Biodiversity Ecosystem Multifunctionality Relationships in R
# Byrnes et al. (2014), Investigating the relationship between biodiversity and ecosystem multifunctionality: challenges and solutions. Methods in Ecology and Evolution, 5: 111???124. doi: 10.1111/2041-210X.12143
# multifunc code from "https://github.com/jebyrnes/multifunc" / https://travis-ci.org/jebyrnes/multifunc
# PDF: http://binarystore.wiley.com/store/10.1111/2041-210X.12143/asset/supinfo/mee312143-sup-0001-DataS1-S2.pdf?v=1&s=753d0de432da50bc6e00515518c401dfa0138b86
#                                                                                                                                        #
# Developed by Aaron Ramus (aaron.ramus@gmail.com). Last updated 21 March 2018.                                                          #
# Edited 21 March 2018 Marjolijn Christianen,                                                                                                                                   #
##########################################################################################################################################


# Clear field
rm(list=ls(all.names=T))

# update R to the latest version

# install required packages
# install.packages("VGAM")
# install.packages("AICcmodavg") # Mazerolle MJ (2016) AICcmodavg: Model selection and multimodel inference based on (Q)AIC(c). R package version 2.0-4. https://cran.r-project.org/web/packages/AICcmodavg/ index.html.
# install.packages("broom")
# install.packages("ggplot2")
# install.packages("devtools") # needed to load multifunc package. This can be found on github, and installed using the devtools library.
# install_github("multifunc", username="jebyrnes")
# install.packages("nls2") # nonlinear least squares for fitting null, linear, log, hyperbolic, and power relationships for each response 
# # Grothendieck G (2013) nls2: Non-linear regression with brute force. R package version0.2. https://cran.r-project.org/web/packages/nls2/index.html.
# install.packages("reshape2")
# install.packages("gridExtra")

# Load required libraries
library(VGAM)
library(AICcmodavg)
library(broom)
library(ggplot2)
library(devtools)
library(multifunc)
library(nls2)
library(reshape2)
# library(gridExtra)

# set work drive
setwd("~/OneDrive - WageningenUR/Papers/XXXX_Christianen_ecosystem services/R_analysis/EMF_script_invFSxfunc")
##########################################################################################################################################

# Read in data
meanPlots <- read.csv("DatasetS1.csv", sep=";")
names(meanPlots)
colnames(meanPlots)[1] <- "plot" 

# Define ID variables
idVars <- c("plot", "d_scen")

# Select predictor variable
predictor <- "TtA"
predictorName <- "Aboveground seagrass biomass (g DW m-2)"

# rename ep.in.Tax to ep.in
colnames(meanPlots)[9] <- "ep.in"    
# Add extra column with ep.in.Tax which is sum of # taxa epifauna, # taxa infauna, # taxa fish
meanPlots$ep.in.Tax <- meanPlots$ep.in+meanPlots$fish.Spec
# Add extra column with Hs.covFlip which is the re-normalized Hs.cov so that everything is on the same  sign-scale (e.g. the maximum level of a function is the "best" function), in this case buffering to invasive species 
meanPlots$Hs.covFlip<- -1*meanPlots$Hs.cov +max(meanPlots$Hs.cov, na.rm=TRUE)
# Add extra column with Hs.bin which is flipped binary version of Hs cover: classification in not invaded (0) and invaded (1) by invasive species 
# meanPlots$Hs.binFlip <- ifelse(meanPlots$Hs.cov== "0", paste("11"), paste("0")) #alternative to flip binairy data

# Select response variables to be included in analysis
mfVars <- qw(
            #TtL,
            #TtB, 
            #Tt.Ll,
            #TtLpr,
            TtL.prN,
            #Sfrac.dec,
            Krate.dec,
            #SedBulkDens,
            #N.TtL,
            #sug.TtZ,
            #sed.TC,
            sed.OC,
            #fish.Spec,
            fish.biom,
            ep.in.Tax, #incl fish
            #ep.in, #excl fish
            #epif.Tax,
            u.SedMob,
            #wav.SedMob,
            #SedGrain.SD50,
            Hs.covFlip
            #Hs.binFlip
            #Tt.nS,
            #inf.Tax
          )

# Calculate average multifunctionality (see Byrnes et al. 2014 Method Ecol Evol for usage of package 'multifunc')
meanPlots <- cbind(meanPlots, getStdAndMeanFunctions(meanPlots, mfVars))

# Create data frame for fits and plots
mfData <- meanPlots[,c(idVars, eval(predictor), mfVars, "meanFunction")]

# Calculate multifunctionality thresholds (again, see Byrnes et al. 2014 Method Ecol Evol for usage of package 'multifunc')
mfThresh <- getFuncsMaxed(meanPlots, mfVars, threshmin=0.10, threshmax=0.90, threshstep=.1, maxN=9, prepend=c(idVars, eval(predictor)))

# Add thresholds to mfData
mfThresh$thresholds <- as.integer(mfThresh$thresholds*100)
mfData$funcMaxed10 <- subset(mfThresh, thresholds==10)$funcMaxed
mfData$funcMaxed20 <- subset(mfThresh, thresholds==20)$funcMaxed
mfData$funcMaxed30 <- subset(mfThresh, thresholds==30)$funcMaxed
mfData$funcMaxed40 <- subset(mfThresh, thresholds==40)$funcMaxed
mfData$funcMaxed50 <- subset(mfThresh, thresholds==50)$funcMaxed
mfData$funcMaxed60 <- subset(mfThresh, thresholds==60)$funcMaxed
mfData$funcMaxed70 <- subset(mfThresh, thresholds==70)$funcMaxed
mfData$funcMaxed80 <- subset(mfThresh, thresholds==80)$funcMaxed
mfData$funcMaxed90 <- subset(mfThresh, thresholds==90)$funcMaxed

# Replace HS.covFlip with unreflected dissolution for fits and plotting
mfData$Hs.covFlip <- meanPlots$Hs.cov
colnames(mfData)[10] <- "Hs.cov"                                #12 is nr of column Hs.cov

##########################################################################################################################################

# Create function to return information about the fitted nls model
getModFit <- function(x) {
   df.modfit <- data.frame(
      Model=as.character(paste(
         summary(x)$'formula'[2], 
         summary(x)$'formula'[1], 
         summary(x)$'formula'[3])), 
      a=round(tidy(x)$estimate[1], 3), 
      pa=tidy(x)$'p.value'[1], 
      b=tidy(x)$estimate[2], 
      pb=tidy(x)$'p.value'[2], 
      c=tidy(x)$estimate[3], 
      pc=tidy(x)$'p.value'[3], 
      RSE=glance(x)[1], 
      df=glance(x)[8], 
      isConv=glance(x)[2], 
      mTol=glance(x)[3], 
      LL2=glance(x)[4], 
      AIC=glance(x)[5], 
      BIC=glance(x)[6], 
      RSS=glance(x)[7], 
      TSS=sum((xy$y-mean(xy$y))^2),
      fvalue=anova(Ho, x)[2,5],
      pvalue=anova(Ho, x)[2,6]
   )
}

# Create parameter dataframes to fit models with package nls2
a <- data.frame(a=c(-100, -50, -10, -5, -1, -0.5, -0.001, 0, 0.001, 0.5, 1, 5, 10, 50, 100))
b <- data.frame(b=c(rep(-100, 15), rep(-50, 15),  rep(-10, 15),  rep(-5, 15),  rep(-1, 15), rep(-0.5, 15), rep(-0.001, 15),  rep(0, 15), rep(0.001, 15),  rep(0.5, 15), rep(1, 15), rep(5, 15), rep(10, 15), rep(50, 15), rep(100, 15)))      
ab <- data.frame(a=rep(a$a, 15), b=b$b)
c <- data.frame(c=c(rep(-100, 225), rep(-50, 225),  rep(-10, 225),  rep(-5, 225),  rep(-1, 225), rep(-0.5, 225), rep(-0.001, 225),  rep(0, 225), rep(0.001, 225),  rep(0.5, 225), rep(1, 225), rep(5, 225), rep(10, 225), rep(50, 225), rep(100, 225)))      
abc <- data.frame(a=rep(ab$a, 15), b=rep(ab$b, 15), c=c$c)
bc <- ab
ac <- ab
colnames(bc) <- c("b", "c")
colnames(ac) <- c("a", "c")

# Define candidate model formulas
HoFormula <- y ~ a  #null
LmFormula <- y ~ a + b * x  #linear
LogFormula <- y ~ a + b * log(x + 1)  #log
HypFormula <- y ~ a * x/(b + x)  #hyperbolic
PowFormula <- y ~ a + b * x^c  #power

# Create an object to stuff model selection tables into later
cum.nlsTab <- NULL

##########################################################################################################################################

# Fit each candidate model to each function individually. This first one is annotated but the rest are not.

# Select response variable to fit models
response <- "TtL.prN"
responseName <- "Nitrogen uptake rate (g m-2 d-1)" # add,'d-1'

# Create XY data frame to fit models
xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")

# Quick plot
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

# Fit null model
HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)

# Plot fitted null model
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

# Fit linear model
LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

# Fit log model
LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

#  Fit hyperbolic model
HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
#Hyp <- nls2(HypFormula, data=xy, start=c(a=75, b=1), nls.control(maxiter=200))  # can try to fit manually
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

#  Fit power model
PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
#Pow <- nls2(PowFormula, data=xy, start=c(a=0.2, b=1, c=1))  # can try to fit manually
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

# All the candidate models need to be fit succussfully before the below file is sourced 
# Otherwise you will need to create a second version of this file and modify it accordingly
source("3 nlsTab.R")


##########################################################################################################################################

response <- "Krate.dec" 
responseName <- "Decomposition rate (k)"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "sed.OC" 
responseName <- "Sediment organic carbon (%)"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart) 
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

# response <- "fish.Spec" 
# responseName <- "Nursery richness (species m^-2)"
# 
# xy <- mfData[,c(paste(predictor), paste(response))]
# colnames(xy) <- c("x", "y")
# p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
# p
# 
# HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
# Ho <- nls2(HoFormula, data=xy, start=HoStart)
# pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
# p+pHo
# 
# LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
# Lm <- nls2(LmFormula, data=xy, start=LmStart)
# pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
# p+pLm
# 
# LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
# Log <- nls2(LogFormula, data=xy, start=LogStart)
# pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
# p+pLog
# 
# HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
# Hyp <- nls2(HypFormula, data=xy, start=HypStart)
# pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
# p+pHyp
# 
# PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
# Pow <- nls2(PowFormula, data=xy, start=PowStart) 
# pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
# p+pPow
# 
# source("3 nlsTab.R")

##########################################################################################################################################

response <- "fish.biom"
responseName <- "Fisheries production (g DW m^-2)"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=ab, control=nls.control(maxiter=200,warnOnly=TRUE,minFactor=1e-5,tol=1e-10),trace=TRUE)  # can try to fit manually
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
# Pow <- nls2(PowFormula, data=xy, start=PowStart)
Pow <- nls2(PowFormula, data=xy, start=abc, control=nls.control(maxiter=200,warnOnly=TRUE,minFactor=1e-5,tol=1e-10),trace=TRUE)  # can try to fit manually
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")
##########################################################################################################################################

response <- "ep.in.Tax"  
responseName <- "Macrobenthic richness (taxa m^-2)"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
#Hyp <- nls2(HypFormula, data=xy, start=HypStart)
Hyp <- nls2(HypFormula, data=xy, start=c(a=4, b=1), nls.control(maxiter=200))  # can try to fit manually
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "u.SedMob"
responseName <- "Threshold shear velocity (m s'^-1,')"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart) 
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

# response <- "wav.SedMob"   
# responseName <- "Sediment stabilisation waves (speed where sediment mobile (cm s^-1)"
# 
# xy <- mfData[,c(paste(predictor), paste(response))]
# colnames(xy) <- c("x", "y")
# p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
# p
# 
# HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
# Ho <- nls2(HoFormula, data=xy, start=HoStart)
# pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
# p+pHo
# 
# LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
# Lm <- nls2(LmFormula, data=xy, start=LmStart)
# pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
# p+pLm
# 
# LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
# Log <- nls2(LogFormula, data=xy, start=LogStart)
# pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
# p+pLog
# 
# HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
# Hyp <- nls2(HypFormula, data=xy, start=HypStart)
# pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
# p+pHyp
# 
# PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
# Pow <- nls2(PowFormula, data=xy, start=abc, control=nls.control(maxiter=200,warnOnly=TRUE,minFactor=1e-5,tol=1e-10),trace=TRUE)  # can try to fit manually
# pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
# p+pPow
# 
# source("3 nlsTab.R")

##########################################################################################################################################

response <- "Hs.cov"
responseName <- "Invasive species cover (%)"
# response <- "Hs.bin"
# responseName <- "Occurance of invasive seagrass"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
#Pow <- nls2(PowFormula, data=xy, start=PowStart) 
Pow <- nls2(PowFormula, data=xy, start=abc, control=nls.control(maxiter=49),trace=TRUE)  # can try to fit manually
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "meanFunction"
responseName <- "Multifunctionality (%)"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
#Pow <- nls2(PowFormula, data=xy, start=PowStart) 
Pow <- nls2(PowFormula, data=xy, start=abc, control=nls.control(maxiter=49),trace=TRUE)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed10"
responseName <- "Number of functions above 10% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=abc, control=nls.control(maxiter=200,warnOnly=TRUE,minFactor=1e-5,tol=1e-10),trace=TRUE)  # can try to fit manually
#Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed20"
responseName <- "Number of functions above 20% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
#Pow <- nls2(PowFormula, data=xy, start=PowStart)
Pow <- nls2(PowFormula, data=xy, start=c(a=0.2, b=7, c=1))
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed30"
responseName <- "Number of functions above 30% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed40"
responseName <- "Number of functions above 40% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed50"
responseName <- "Number of functions above 50% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed60"
responseName <- "Number of functions above 60% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed70"
responseName <- "Number of functions above 70% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
Hyp <- nls2(HypFormula, data=xy, start=HypStart)
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed80"
responseName <- "Number of functions above 80% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
#Hyp <- nls2(HypFormula, data=xy, start=HypStart)
Hyp <- nls2(HypFormula, data=xy, start=ab, nls.control(maxiter=200)) 
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

response <- "funcMaxed90"
responseName <- "Number of functions above 90% threshold"

xy <- mfData[,c(paste(predictor), paste(response))]
colnames(xy) <- c("x", "y")
p <- qplot(x, y, data=xy) + labs(x=paste(eval(predictorName)), y=paste(eval(responseName)))
p

HoStart <- nls2(HoFormula, data=xy, start=a, algorithm="brute-force")
Ho <- nls2(HoFormula, data=xy, start=HoStart)
pHo <- stat_function(fun=function(x){coef(Ho)[1]+0*x}, aes(col="Null"))
p+pHo

LmStart <- nls2(LmFormula, data=xy, start=ab, algorithm="brute-force")
Lm <- nls2(LmFormula, data=xy, start=LmStart)
pLm <-stat_function(fun=function(x){coef(Lm)[1]+coef(Lm)[2]*x}, aes(col="Linear"))
p+pLm

LogStart <- nls2(LogFormula, data=xy, start=ab, algorithm="brute-force")
Log <- nls2(LogFormula, data=xy, start=LogStart)
pLog <- stat_function(fun=function(x){coef(Log)[1]+coef(Log)[2]*log(x+1)}, aes(col="Log"))
p+pLog

HypStart <- nls2(HypFormula, data=xy, start=ab, algorithm="brute-force")
#Hyp <- nls2(HypFormula, data=xy, start=HypStart)
Hyp <- nls2(HypFormula, data=xy, start=ab, nls.control(maxiter=200)) 
pHyp <- stat_function(fun=function(x){coef(Hyp)[1]*x/(coef(Hyp)[2]+x)}, aes(col="Hyperbolic"))
p+pHyp

PowStart <- nls2(PowFormula, data=xy, start=abc, algorithm="brute-force")
Pow <- nls2(PowFormula, data=xy, start=PowStart)
pPow <- stat_function(fun=function(x){coef(Pow)[1]+coef(Pow)[2]*x^coef(Pow)[3]}, aes(col="Power"))
p+pPow

source("3 nlsTab.R")

##########################################################################################################################################

# Write model selection table
write.csv(cum.nlsTab, "4 model selection table.csv")

##########################################################################################################################################

# Melt data for plotting
melt <- melt(mfData, id.vars=c(idVars, paste(predictor)))
melt$d_scen <- factor(melt$d_scen)


# Give pretty names for plotting
melt$variable2 <- melt$variable
# melt$variable2 <- factor(melt$variable2)
melt$variable2 <- ifelse(melt$variable=="TtL.prN", paste("paste('Nitrogen uptake rate (g m'^-2,')')"), paste(melt$variable2)) #! add  ,'d-1' behind 2
melt$variable2 <- ifelse(melt$variable=="Krate.dec", paste("paste('Decomposition rate (k)')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="sed.OC", paste("paste('Sediment organic carbon (% DW)')"), paste(melt$variable2))
#melt$variable2 <- ifelse(melt$variable=="fish.Spec", paste("paste('Nursery richness (taxa m'^-2,')')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="fish.biom", paste("paste('Fish biomass (g DW m'^-2,')')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="ep.in.Tax", paste("paste('Macrofauna richness (taxa m'^-2,')')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="u.SedMob", paste("paste('Threshold shear velocity (m s'^-1,')')"), paste(melt$variable2))
#melt$variable2 <- ifelse(melt$variable=="w.SedMob", paste("paste('threshold shear velocity (m s'^-1,')')"), paste(melt$variable2))
# melt$variable2 <- ifelse(melt$variable=="Hs.bin", paste("paste('Occurance of invasive seagrass"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="Hs.cov", paste("paste('Invasive species cover (%)')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="meanFunction", paste("paste('Multifunctionality (%)')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed10", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed20", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed30", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed40", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed50", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed60", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed70", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed80", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- ifelse(melt$variable=="funcMaxed90", paste("paste('Number of functions'>= 'threshold')"), paste(melt$variable2))
melt$variable2 <- factor(melt$variable2, levels=c("paste('Nitrogen uptake rate (g m'^-2,')')", # ,'d-1'
                                                  "paste('Decomposition rate (k)')",
                                                  "paste('Sediment organic carbon (% DW)')",
                                                 # "paste('Nursery richness (taxa m'^-2,')')",
                                                  "paste('Fish biomass (g DW m'^-2,')')",
                                                  "paste('Macrofauna richness (taxa m'^-2,')')",
                                                  "paste('Threshold shear velocity (m s'^-1,')')",
                                                 # "paste('Threshold shear velocity (m s'^-1,')')",
                                                  "paste('Invasive species cover (%)')",
                                                  "paste('Multifunctionality (%)')", 
                                                  "paste('Number of functions'>= 'threshold')")) 

# Subset functions for which you plan to add lines
TtL.prN <- subset(melt, melt$variable2=="paste('Nitrogen uptake rate (g m'^-2,')')") #,'d-1'
TtL.prN$variable2 <- factor(TtL.prN$variable2)
Krate.dec <- subset(melt, melt$variable2=="paste('Decomposition rate (k)')")
Krate.dec$variable2 <- factor(Krate.dec$variable2)
sed.OC <- subset(melt, melt$variable2=="paste('Sediment organic carbon (% DW)')")
sed.OC$variable2 <- factor(sed.OC$variable2)
# fish.Spec <- subset(melt, melt$variable2=="paste('Nursery richness (taxa m'^-2,')')")
# fish.Spec$variable2 <- factor(fish.Spec$variable2)
fish.biom <- subset(melt, melt$variable2=="paste('Fish biomass (g DW m'^-2,')')")
fish.biom$variable2 <- factor(fish.biom$variable2)
ep.in.Tax <- subset(melt, melt$variable2=="paste('Macrofauna richness (taxa m'^-2,')')")
ep.in.Tax$variable2 <- factor(ep.in.Tax$variable2)
u.SedMob <- subset(melt, melt$variable2=="paste('Threshold shear velocity (m s'^-1,')')")
u.SedMob$variable2 <- factor(u.SedMob$variable2)
# wav.SedMob <- subset(melt, melt$variable2=="paste('Threshold shear velocity (m s'^-1,')')")
# wav.SedMob$variable2 <- factor(wav.SedMob$variable2)
Hs.cov <- subset(melt, melt$variable2=="paste('Invasive species cover (%)')") #not significant
Hs.cov$variable2 <- factor(Hs.cov$variable2)
meanFunction <- subset(melt, melt$variable2=="paste('Multifunctionality (%)')")
meanFunction$variable2 <- factor(meanFunction$variable2)

# Subset to plot points vs. threshold lines
funcs <- subset(melt, melt$variable2!="paste('Number of functions'>= 'threshold')")
funcs$variable2 <- factor(funcs$variable2)
funcs$d_scen <- factor(funcs$d_scen)
tholds <- subset(melt, melt$variable2=="paste('Number of functions'>= 'threshold')")
tholds$variable2 <- factor(tholds$variable2)

# Subset to plot individual thresholds
tholds$thresholds <- as.character(tholds$variable)
tholds$thresholds <- as.numeric(with(tholds, gsub("funcMaxed", "", tholds$thresholds)))
t10 <- subset(tholds, thresholds == 10)
t20 <- subset(tholds, thresholds == 20)
t30 <- subset(tholds, thresholds == 30)
t40 <- subset(tholds, thresholds == 40)
t50 <- subset(tholds, thresholds == 50)
t60 <- subset(tholds, thresholds == 60)
t70 <- subset(tholds, thresholds == 70)
t80 <- subset(tholds, thresholds == 80)
t90 <- subset(tholds, thresholds == 90)

##########################################################################################################################################

# Generate plot
#### To Do:
#line 998 below ('facet_wrap...) gives error when 'labeller=label_parsed' is included so I leave it out for now - needs fixing

fig2 <-       ggplot(aes(x=TtA, y=value), data=melt) + 
   facet_wrap(~variable2, scales="free", ncol=3, labeller=label_parsed)+                   # adjusted to 4 columns wide/ label=labels)
   stat_function(data=TtL.prN, fun=function(x){4.868*x/(x+26.342)}, size=0.25, lty=1, alpha=1) +               #Sed?
   theme_bw(base_size=6, base_family="Helvetica") +                                       #arial?
   geom_point(data=funcs, aes(shape=factor(d_scen), fill=factor(d_scen)), stroke=0.25, color="black") + 
      scale_shape_manual(values=c(21, 24, 22)) +                                    #adjust to 3 treatments and adjust numbers
      scale_fill_manual(values=c("purple", "red", "yellow")) +                      #adjust to 3 treatments and adjust numbers
      guides(shape=F, fill=F) +
   #stat_function(data=TtL.prN, fun=function(x){4.868*x/(x+26.342)}, size=0.25) + #copied and moved this line up to check if this is causing error
   stat_function(data=Krate.dec, fun=function(x){0.003+0.000031844*x}, size=0.25) + 
   stat_function(data=sed.OC, fun=function(x){0.586+0.077*log(x+1)}, size=0.25) + 
   # stat_function(data=fish.Spec, fun=function(x){5.097+0.093*x}, size=0.25) + 
   stat_function(data=fish.biom, fun=function(x){-1.806*x/(x-520.123)}, size=0.25) +  #{-0.875*x/(x-243.452)}
   stat_function(data=ep.in.Tax, fun=function(x){7.205+2.707*log(x+1)}, size=0.25) +        #new: ep+infauna+fish, old:epi+infauna {5.328+0.027*x}
   stat_function(data=u.SedMob, fun=function(x){5.651+2.661*x^0.602}, size=0.25) +
   # stat_function(data=wav.SedMob, fun=function(x){7.936+0.0003850182*x^2.061}, size=0.25) + 
   stat_function(data=Hs.cov, fun=function(x){10.023-0.055*x}, lty=2, size=0.25) +           #not sign. Hs.covFlip? 
   stat_function(data=meanFunction, fun=function(x){0.255+0.04*x^0.545}, size=0.25) + 
   stat_function(fun=function(x){6.600}, data=t10, aes(colour=thresholds)) +            
   stat_function(fun=function(x){4.868+0.145*x^0.556}, data=t20, aes(colour=thresholds)) +
   stat_function(fun=function(x){2.943+0.488*x^0.436}, data=t30, aes(colour=thresholds)) +
   stat_function(fun=function(x){1.748+1.005*log(x+1)}, data=t40, aes(colour=thresholds)) +
   stat_function(fun=function(x){1.026+1.133*log(x+1)}, data=t50, aes(colour=thresholds)) +
   stat_function(fun=function(x){-0.37+1.431*log(x+1)}, data=t60, aes(colour=thresholds)) +
   stat_function(fun=function(x){7.605*x/(28.76+x)}, data=t70, aes(colour=thresholds)) + 
   stat_function(fun=function(x){11.617*x/(114.235+x)}, data=t80, aes(colour=thresholds)) +
   stat_function(fun=function(x){0.299+0.04*x}, data=t90, aes(colour=thresholds)) +
   xlab("\nAboveground seagrass biomass (g m'^-2,')") +                         
   theme(
      panel.grid=element_blank(),
      aspect.ratio=1,
      panel.background=element_rect(fill=NA),
      line=element_line(size=0.25),
      axis.title.y=element_blank(),
      axis.title.x=element_text(size=unit(6.5, "pt")),
      axis.ticks.length=unit(-0.05, "cm"),
      axis.text.x= element_text(margin = margin(t = 3)),
      axis.text.y=element_text(margin = margin (r=3)),
      strip.background=element_rect(fill=NA, colour=NA),
      axis.text=element_text(colour="black", size=unit(6, "pt")),
      strip.text=element_text(size=unit(6, "pt")),
      panel.spacing=unit(0.5, "lines"),
      panel.border=element_rect(fill=NA, color=NA), 
      axis.line=element_line(size=0.25),
      legend.justification="top",
      legend.position=c(0.9225, 0.08),
      legend.direction="horizontal",
      legend.background=element_rect(fill=NA),
      legend.margin=unit(0, "cm"),
      legend.spacing=unit(0, "cm"),
      legend.text=element_text(size=unit(6, "pt")),
      legend.key=element_rect(fill = NA, colour = NA),
      legend.key.size = unit(1, "cm"),
      legend.key.height =  unit(0.15, "cm"),
      legend.key.width =   unit(0.2625, "cm"),
      legend.title=element_text(face="plain", size=unit(6, "pt"))) +
   scale_color_gradientn(name="Threshold (%)", 
      colours=rev(rainbow(5)), # 5
      guide=guide_colourbar(title.position="top"), breaks=c(10, 50, 90))
# fig2 # error when plotting, however plot will print correctly using script below
ggsave(fig2, file="5 figure.png", width=4.5, height=5.1, dpi=1200)

##################################################################################

